/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Namespace
    Foam::debug

Description
    Namespace for handling debugging switches.

SourceFiles
    debug.C

\*---------------------------------------------------------------------------*/

#ifndef debug_H
#define debug_H

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward declarations
class dictionary;
class Istream;
class Ostream;
class simpleRegIOobject;
class simpleObjectRegistry;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace debug
{
    //- The central control dictionary, the contents of which are either
    //- taken directly from the FOAM_CONTROLDICT environment variable,
    //- or generated by merging the contents of user/group/other files.
    //
    //  - ~/.OpenFOAM/{PROJECT_API}/controlDict
    //  - ~/.OpenFOAM/controlDict
    //  - $WM_PROJECT_SITE/{PROJECT_API}/etc/controlDict
    //  - $WM_PROJECT_SITE/etc/controlDict
    //  - $WM_PROJECT_DIR/etc/controlDict
    //
    //  Where {PROJECT_API} corresponds to the foamVersion::api value.
    //
    //  \sa Foam::findEtcFile()
    dictionary& controlDict();

    //- The DebugSwitches sub-dictionary in the central controlDict(s).
    dictionary& debugSwitches();

    //- The InfoSwitches sub-dictionary in the central controlDict(s).
    dictionary& infoSwitches();

    //- The OptimisationSwitches sub-dictionary in the central controlDict(s).
    dictionary& optimisationSwitches();

    //- Lookup debug switch or add default value.
    int debugSwitch(const char* name, const int deflt = 0);

    //- Lookup info switch or add default value.
    int infoSwitch(const char* name, const int deflt = 0);

    //- Lookup optimisation switch or add default value.
    int optimisationSwitch(const char* name, const int deflt = 0);

    //- Lookup optimisation switch or add default value.
    float floatOptimisationSwitch(const char* name, const float deflt = 0);

    //- Internal function to lookup a sub-dictionary from controlDict.
    dictionary& switchSet(const char* subDictName, dictionary*& subDictPtr);


    // Registered debug switches

        //- Register debug switch read/write object
        void addDebugObject(const char* name, simpleRegIOobject* obj);

        //- Register info switch read/write object
        void addInfoObject(const char* name, simpleRegIOobject* obj);

        //- Register optimisation switch read/write object
        void addOptimisationObject(const char* name, simpleRegIOobject* obj);

        //- Register DimensionSets read/write object
        void addDimensionSetObject(const char* name, simpleRegIOobject* obj);

        //- Register DimensionedConstant read/write object
        void addDimensionedConstantObject(const char* name, simpleRegIOobject*);


        //- Access to registered DebugSwitch objects
        simpleObjectRegistry& debugObjects();

        //- Access to registered InfoSwitch objects
        simpleObjectRegistry& infoObjects();

        //- Access to registered OptimisationSwitch objects
        simpleObjectRegistry& optimisationObjects();

        //- Access to registered DimensionSets objects
        simpleObjectRegistry& dimensionSetObjects();

        //- Access to registered DimensionedConstants objects
        simpleObjectRegistry& dimensionedConstantObjects();


    // List switches

    //- List debug/info/optimisation switches
    void listSwitches(const bool unset = false);

    //- List debug switches
    void listDebugSwitches(const bool unset = false);

    //- List info switches
    void listInfoSwitches(const bool unset = false);

    //- List optimisation switches
    void listOptimisationSwitches(const bool unset = false);


    //- List registered debug/info/optimisation switches
    void listRegisteredSwitches(const bool unset = false);

    //- List debug switches
    void listRegisteredDebugSwitches(const bool unset = false);

    //- List info switches
    void listRegisteredInfoSwitches(const bool unset = false);

    //- List optimisation switches
    void listRegisteredOptimisationSwitches(const bool unset = false);

} // End namespace debug


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
