#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# This script read cipher suite list csv file [1] and prints out id
# and name of black listed cipher suites.  The output is used by
# src/ssl.cc.
#
# [1] http://www.iana.org/assignments/tls-parameters/tls-parameters-4.csv
# [2] http://www.iana.org/assignments/tls-parameters/tls-parameters.xhtml

import re
import sys
import csv

# From RFC 7540
blacklist = [
    'TLS_NULL_WITH_NULL_NULL',
    'TLS_RSA_WITH_NULL_MD5',
    'TLS_RSA_WITH_NULL_SHA',
    'TLS_RSA_EXPORT_WITH_RC4_40_MD5',
    'TLS_RSA_WITH_RC4_128_MD5',
    'TLS_RSA_WITH_RC4_128_SHA',
    'TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5',
    'TLS_RSA_WITH_IDEA_CBC_SHA',
    'TLS_RSA_EXPORT_WITH_DES40_CBC_SHA',
    'TLS_RSA_WITH_DES_CBC_SHA',
    'TLS_RSA_WITH_3DES_EDE_CBC_SHA',
    'TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA',
    'TLS_DH_DSS_WITH_DES_CBC_SHA',
    'TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA',
    'TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA',
    'TLS_DH_RSA_WITH_DES_CBC_SHA',
    'TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA',
    'TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA',
    'TLS_DHE_DSS_WITH_DES_CBC_SHA',
    'TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA',
    'TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA',
    'TLS_DHE_RSA_WITH_DES_CBC_SHA',
    'TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA',
    'TLS_DH_anon_EXPORT_WITH_RC4_40_MD5',
    'TLS_DH_anon_WITH_RC4_128_MD5',
    'TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA',
    'TLS_DH_anon_WITH_DES_CBC_SHA',
    'TLS_DH_anon_WITH_3DES_EDE_CBC_SHA',
    'TLS_KRB5_WITH_DES_CBC_SHA',
    'TLS_KRB5_WITH_3DES_EDE_CBC_SHA',
    'TLS_KRB5_WITH_RC4_128_SHA',
    'TLS_KRB5_WITH_IDEA_CBC_SHA',
    'TLS_KRB5_WITH_DES_CBC_MD5',
    'TLS_KRB5_WITH_3DES_EDE_CBC_MD5',
    'TLS_KRB5_WITH_RC4_128_MD5',
    'TLS_KRB5_WITH_IDEA_CBC_MD5',
    'TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA',
    'TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA',
    'TLS_KRB5_EXPORT_WITH_RC4_40_SHA',
    'TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5',
    'TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5',
    'TLS_KRB5_EXPORT_WITH_RC4_40_MD5',
    'TLS_PSK_WITH_NULL_SHA',
    'TLS_DHE_PSK_WITH_NULL_SHA',
    'TLS_RSA_PSK_WITH_NULL_SHA',
    'TLS_RSA_WITH_AES_128_CBC_SHA',
    'TLS_DH_DSS_WITH_AES_128_CBC_SHA',
    'TLS_DH_RSA_WITH_AES_128_CBC_SHA',
    'TLS_DHE_DSS_WITH_AES_128_CBC_SHA',
    'TLS_DHE_RSA_WITH_AES_128_CBC_SHA',
    'TLS_DH_anon_WITH_AES_128_CBC_SHA',
    'TLS_RSA_WITH_AES_256_CBC_SHA',
    'TLS_DH_DSS_WITH_AES_256_CBC_SHA',
    'TLS_DH_RSA_WITH_AES_256_CBC_SHA',
    'TLS_DHE_DSS_WITH_AES_256_CBC_SHA',
    'TLS_DHE_RSA_WITH_AES_256_CBC_SHA',
    'TLS_DH_anon_WITH_AES_256_CBC_SHA',
    'TLS_RSA_WITH_NULL_SHA256',
    'TLS_RSA_WITH_AES_128_CBC_SHA256',
    'TLS_RSA_WITH_AES_256_CBC_SHA256',
    'TLS_DH_DSS_WITH_AES_128_CBC_SHA256',
    'TLS_DH_RSA_WITH_AES_128_CBC_SHA256',
    'TLS_DHE_DSS_WITH_AES_128_CBC_SHA256',
    'TLS_RSA_WITH_CAMELLIA_128_CBC_SHA',
    'TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA',
    'TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA',
    'TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA',
    'TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA',
    'TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA',
    'TLS_DHE_RSA_WITH_AES_128_CBC_SHA256',
    'TLS_DH_DSS_WITH_AES_256_CBC_SHA256',
    'TLS_DH_RSA_WITH_AES_256_CBC_SHA256',
    'TLS_DHE_DSS_WITH_AES_256_CBC_SHA256',
    'TLS_DHE_RSA_WITH_AES_256_CBC_SHA256',
    'TLS_DH_anon_WITH_AES_128_CBC_SHA256',
    'TLS_DH_anon_WITH_AES_256_CBC_SHA256',
    'TLS_RSA_WITH_CAMELLIA_256_CBC_SHA',
    'TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA',
    'TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA',
    'TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA',
    'TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA',
    'TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA',
    'TLS_PSK_WITH_RC4_128_SHA',
    'TLS_PSK_WITH_3DES_EDE_CBC_SHA',
    'TLS_PSK_WITH_AES_128_CBC_SHA',
    'TLS_PSK_WITH_AES_256_CBC_SHA',
    'TLS_DHE_PSK_WITH_RC4_128_SHA',
    'TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA',
    'TLS_DHE_PSK_WITH_AES_128_CBC_SHA',
    'TLS_DHE_PSK_WITH_AES_256_CBC_SHA',
    'TLS_RSA_PSK_WITH_RC4_128_SHA',
    'TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA',
    'TLS_RSA_PSK_WITH_AES_128_CBC_SHA',
    'TLS_RSA_PSK_WITH_AES_256_CBC_SHA',
    'TLS_RSA_WITH_SEED_CBC_SHA',
    'TLS_DH_DSS_WITH_SEED_CBC_SHA',
    'TLS_DH_RSA_WITH_SEED_CBC_SHA',
    'TLS_DHE_DSS_WITH_SEED_CBC_SHA',
    'TLS_DHE_RSA_WITH_SEED_CBC_SHA',
    'TLS_DH_anon_WITH_SEED_CBC_SHA',
    'TLS_RSA_WITH_AES_128_GCM_SHA256',
    'TLS_RSA_WITH_AES_256_GCM_SHA384',
    'TLS_DH_RSA_WITH_AES_128_GCM_SHA256',
    'TLS_DH_RSA_WITH_AES_256_GCM_SHA384',
    'TLS_DH_DSS_WITH_AES_128_GCM_SHA256',
    'TLS_DH_DSS_WITH_AES_256_GCM_SHA384',
    'TLS_DH_anon_WITH_AES_128_GCM_SHA256',
    'TLS_DH_anon_WITH_AES_256_GCM_SHA384',
    'TLS_PSK_WITH_AES_128_GCM_SHA256',
    'TLS_PSK_WITH_AES_256_GCM_SHA384',
    'TLS_RSA_PSK_WITH_AES_128_GCM_SHA256',
    'TLS_RSA_PSK_WITH_AES_256_GCM_SHA384',
    'TLS_PSK_WITH_AES_128_CBC_SHA256',
    'TLS_PSK_WITH_AES_256_CBC_SHA384',
    'TLS_PSK_WITH_NULL_SHA256',
    'TLS_PSK_WITH_NULL_SHA384',
    'TLS_DHE_PSK_WITH_AES_128_CBC_SHA256',
    'TLS_DHE_PSK_WITH_AES_256_CBC_SHA384',
    'TLS_DHE_PSK_WITH_NULL_SHA256',
    'TLS_DHE_PSK_WITH_NULL_SHA384',
    'TLS_RSA_PSK_WITH_AES_128_CBC_SHA256',
    'TLS_RSA_PSK_WITH_AES_256_CBC_SHA384',
    'TLS_RSA_PSK_WITH_NULL_SHA256',
    'TLS_RSA_PSK_WITH_NULL_SHA384',
    'TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256',
    'TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256',
    'TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256',
    'TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256',
    'TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256',
    'TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256',
    'TLS_EMPTY_RENEGOTIATION_INFO_SCSV',
    'TLS_ECDH_ECDSA_WITH_NULL_SHA',
    'TLS_ECDH_ECDSA_WITH_RC4_128_SHA',
    'TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA',
    'TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA',
    'TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA',
    'TLS_ECDHE_ECDSA_WITH_NULL_SHA',
    'TLS_ECDHE_ECDSA_WITH_RC4_128_SHA',
    'TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA',
    'TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA',
    'TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA',
    'TLS_ECDH_RSA_WITH_NULL_SHA',
    'TLS_ECDH_RSA_WITH_RC4_128_SHA',
    'TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA',
    'TLS_ECDH_RSA_WITH_AES_128_CBC_SHA',
    'TLS_ECDH_RSA_WITH_AES_256_CBC_SHA',
    'TLS_ECDHE_RSA_WITH_NULL_SHA',
    'TLS_ECDHE_RSA_WITH_RC4_128_SHA',
    'TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA',
    'TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA',
    'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA',
    'TLS_ECDH_anon_WITH_NULL_SHA',
    'TLS_ECDH_anon_WITH_RC4_128_SHA',
    'TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA',
    'TLS_ECDH_anon_WITH_AES_128_CBC_SHA',
    'TLS_ECDH_anon_WITH_AES_256_CBC_SHA',
    'TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA',
    'TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA',
    'TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA',
    'TLS_SRP_SHA_WITH_AES_128_CBC_SHA',
    'TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA',
    'TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA',
    'TLS_SRP_SHA_WITH_AES_256_CBC_SHA',
    'TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA',
    'TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA',
    'TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256',
    'TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384',
    'TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256',
    'TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384',
    'TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256',
    'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384',
    'TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256',
    'TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384',
    'TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256',
    'TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384',
    'TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256',
    'TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384',
    'TLS_ECDHE_PSK_WITH_RC4_128_SHA',
    'TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA',
    'TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA',
    'TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA',
    'TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256',
    'TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384',
    'TLS_ECDHE_PSK_WITH_NULL_SHA',
    'TLS_ECDHE_PSK_WITH_NULL_SHA256',
    'TLS_ECDHE_PSK_WITH_NULL_SHA384',
    'TLS_RSA_WITH_ARIA_128_CBC_SHA256',
    'TLS_RSA_WITH_ARIA_256_CBC_SHA384',
    'TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256',
    'TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384',
    'TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256',
    'TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384',
    'TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256',
    'TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384',
    'TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256',
    'TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384',
    'TLS_DH_anon_WITH_ARIA_128_CBC_SHA256',
    'TLS_DH_anon_WITH_ARIA_256_CBC_SHA384',
    'TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256',
    'TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384',
    'TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256',
    'TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384',
    'TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256',
    'TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384',
    'TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256',
    'TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384',
    'TLS_RSA_WITH_ARIA_128_GCM_SHA256',
    'TLS_RSA_WITH_ARIA_256_GCM_SHA384',
    'TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256',
    'TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384',
    'TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256',
    'TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384',
    'TLS_DH_anon_WITH_ARIA_128_GCM_SHA256',
    'TLS_DH_anon_WITH_ARIA_256_GCM_SHA384',
    'TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256',
    'TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384',
    'TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256',
    'TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384',
    'TLS_PSK_WITH_ARIA_128_CBC_SHA256',
    'TLS_PSK_WITH_ARIA_256_CBC_SHA384',
    'TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256',
    'TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384',
    'TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256',
    'TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384',
    'TLS_PSK_WITH_ARIA_128_GCM_SHA256',
    'TLS_PSK_WITH_ARIA_256_GCM_SHA384',
    'TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256',
    'TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384',
    'TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256',
    'TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384',
    'TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384',
    'TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384',
    'TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384',
    'TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384',
    'TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256',
    'TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384',
    'TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256',
    'TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384',
    'TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256',
    'TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384',
    'TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256',
    'TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384',
    'TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256',
    'TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384',
    'TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256',
    'TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384',
    'TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256',
    'TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384',
    'TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256',
    'TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384',
    'TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384',
    'TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384',
    'TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384',
    'TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256',
    'TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384',
    'TLS_RSA_WITH_AES_128_CCM',
    'TLS_RSA_WITH_AES_256_CCM',
    'TLS_RSA_WITH_AES_128_CCM_8',
    'TLS_RSA_WITH_AES_256_CCM_8',
    'TLS_PSK_WITH_AES_128_CCM',
    'TLS_PSK_WITH_AES_256_CCM',
    'TLS_PSK_WITH_AES_128_CCM_8',
    'TLS_PSK_WITH_AES_256_CCM_8',
]

ciphers = []
found = set()
for hl, name, _, _, _ in csv.reader(sys.stdin):
    if name not in blacklist:
        continue

    found.add(name)

    high, low = hl.split(',')

    id = high + low[2:] + 'u'
    ciphers.append((id, name))

print('''\
enum {''')

for id, name in ciphers:
    print('{} = {},'.format(name, id))

print('''\
};
''')

for id, name in ciphers:
    print('''\
case {}:'''.format(name))

if len(found) != len(blacklist):
    print('{} found out of {}; not all cipher was found: {}'.format(
        len(found), len(blacklist),
        found.symmetric_difference(blacklist)))
